COMMENT
Consider a frequency pattern, f(t), that represents a spike train
t_i where t_0 is the time of the first spike and the interval
dt_i = t_{i+1} - t_{i}
        
Thus
t_{i+1} = t_i + dt_i
and
dt_i = 1 over { f(t_i + dt_i/2 }
Note that the value of the interval is the reciprocal of the frequency
at the midpoint of the interval. A second order approximation is
                
dt_i = f(t_i)/(f^2(t_i) + .5f'(t_i))
                        
For some particularly simple forms of f it may be possible to
analytically solve for dt_i but such fastidiousness must be weighed
with respect to the earlier transition between a continuous function
of f and a discrete sequence of events, t_i.
 
Consider a generator that produces a ramp frequency starting at
time, tstart, with frequency, fmax, and which decreases to fmin
linearly at time, tstop. We resolve the ambiguity about whether
to allow one more spike after tstop by choosing, no.

We get the stream started with an initial block that ensures that the
NET_RECEIVE block gets a self-event at time tstart.

ENDCOMMENT

NEURON {
	POINT_PROCESS FrequencyRamp
	RANGE tstart, tstop, fstart, fstop, noise
}

PARAMETER {
	tstart = 0 (ms)
	tstop = 100 (ms)
	fstart = 100 (hz)
	fstop = 1 (hz)
	noise = 0 <0,1>
}

INITIAL {
	if (tstart >= 0 && tstart < tstop) {
		net_send(tstart, 1)
	}
}

NET_RECEIVE(w) {
	LOCAL x, d, f, df
	if (flag == 1) {
		net_event(t)
		x = (t - tstart)/(tstop - tstart)
		f = fstart*(1 - x) + fstop*x
		df = (1000)*(fstart - fstop)/(tstart - tstop)
		x = f*f + .5*df
		if (x > 0) {
			d = (1000)*f/x
			if (noise > 0) {
				d = d*((1-noise) + noise*exprand(1))
			}
			if (d + t <= tstop) {
				net_send(d, 1)
			}
		}
	}
}

